<?php
/**
 * Display content of "allow list" tab on settings page
 * @package Captcha Plus
 * @since   4.1.4
 * @version 1.0.2
 */

if ( ! class_exists( 'Cptch_allowlist' ) ) {
	if ( ! class_exists( 'WP_List_Table' ) ) {
		require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
	}

	class Cptch_allowlist extends WP_List_Table {
		private
			$disable_list,
			$basename,
			$order_by,
			$per_page,
			$la_info,
			$paged,
			$order,
			$s,
			$date_filter_options,
			$filter_by,
			$last_filtred_by;

		/**
		* Constructor of class
		*/
		function __construct( $plugin_basename, $limit_attempts_info ) {
			global $cptch_options;

			parent::__construct( array(
				'singular'	=> 'IP',
				'plural'	=> 'IP',
				'ajax'		=> true,
				)
			);
			$this->basename		= $plugin_basename;
			$this->la_info		= $limit_attempts_info;

			/**
			* options for filtring
			*/
			$this->date_filter_options = array(
				'all'	=> esc_html__( 'All dates', 'captcha-plus' ),
				'day'	=> esc_html__( 'Last 24 hours', 'captcha-plus' ),
				'week'	=> esc_html__( 'Last week', 'captcha-plus' ),
				'month'	=> esc_html__( 'Last month', 'captcha-plus' ),
				'year'	=> esc_html__( 'Last year', 'captcha-plus' )
			);

			/**
			* keep in mind what was the last filtring option to compare it
			* with the new filtring options and choose the differnt one
			*/
			if (
				! empty( $_POST['cptch_last_filtred_by'] ) &&
				in_array( $_POST['cptch_last_filtred_by'], array_keys( $this->date_filter_options ) )
			) {
				$this->last_filtred_by = $_POST['cptch_last_filtred_by'];
			} else {
				$this->last_filtred_by = 'all';
			}

			if ( ! empty( $_POST['cptch_date_filter'] ) ) {
				$filter_array = array_filter( array_unique( $_POST['cptch_date_filter'] ), array( $this, 'get_date_filter_values' ) );
			}

			/**
			* Due to the first element's key either be 0 or 1, $filter_array[ key( $filter_array ) ] should be used.
			* It gives the ability of taking the first element of the array
			*/
			$this->filter_by = ! empty( $filter_array ) ? $filter_array[ key( $filter_array ) ] : $this->last_filtred_by;

			$this->disable_list = ( 1 == $cptch_options['use_limit_attempts_allowlist'] ) && 'active' == $this->la_info['status'];
		}

		/**
		 * Display content
		 * @return void
		 */
		function display_content() { ?>
			<h1>
                <?php esc_html_e( 'Captcha Plus Allow List', 'captcha-plus' ); ?>
                <?php if ( ! $this->disable_list ) { ?>
                    <form method="post" action="admin.php?page=captcha-allowlist.php" class="cptch_allowlist_add_new" style="display: inline;">
                        <button class="page-title-action" name="cptch_show_allowlist_form" value="on"<?php echo ( isset( $_POST['cptch_add_to_allowlist'] ) ) ? ' style="display: none;"' : ''; ?>><?php esc_html_e( 'Add New', 'captcha-plus' ); ?></button>
                    </form>
                <?php } ?>
            </h1>
			<?php if ( isset( $_SERVER ) ) {
				$sever_vars = array( 'HTTP_X_REAL_IP', 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR' );
				foreach ( $sever_vars as $var ) {
					if ( isset( $_SERVER[ $var ] ) && ! empty( $_SERVER[ $var ] ) ) {
						if ( filter_var( $_SERVER[ $var ], FILTER_VALIDATE_IP ) ) {
							$my_ip = $_SERVER[ $var ];
							break;
						} else { /* if proxy */
							$ip_array = explode( ',', $_SERVER[ $var ] );
							if ( is_array( $ip_array ) && ! empty( $ip_array ) && filter_var( $ip_array[0], FILTER_VALIDATE_IP ) ) {
								$my_ip = $ip_array[0];
								break;
							}
						}
					}
				}
			}

			$this->display_notices();
			$this->prepare_items();

			$limit_attempts_info = $this->get_limit_attempts_info();
			$disabled = $limit_attempts_info['disabled'] ? ' disabled="disabled"' : '';

			if ( ! ( isset( $_REQUEST['cptch_show_allowlist_form'] ) || isset( $_REQUEST['cptch_add_to_allowlist'] ) ) ) { ?>
				<form method="post" action="admin.php?page=captcha-allowlist.php">
					<?php if ( $this->disable_list ) { ?>
						<p><?php echo esc_html__( 'Limit Attempts allow list is activated.', 'captcha-plus' ) .
							' <a href="admin.php?page=limit-attempts-black-and-allow-list.php">' . esc_html__( 'View the allow list', 'captcha-plus' ) . '</a> ' .
							esc_html__( 'or', 'captcha-plus' ) .
							' <a href="admin.php?page=captcha-plus.php">' . esc_html__( 'Enable default captcha allow list', 'captcha-plus' ) . '</a>'; ?></p>
					<?php } else { ?>
						<div>
							<input type="submit" name="cptch_load_limit_attempts_allowlist" class="button" value="<?php esc_html_e( 'Load IP Address(-es)', 'captcha-plus' ); ?>" style="float: left;" <?php echo $disabled; ?> />
							<noscript>
								<div class="alignleft" style="margin: 0 10px;">
									<input type="submit" name="cptch_save_add_ip_form_button" class="button-primary" value="<?php esc_html_e( 'Save changes', 'captcha-plus' ); ?>" />
								</div>
							</noscript>
							<div class="clear"></div>
						</div>
						<div class="bws_info"><?php esc_html_e( 'Load IP addresses from the "Limit Attempts" plugin.', 'captcha-plus' ); ?> <?php echo ! empty( $disabled ) ? $limit_attempts_info['link'] : ''; ?></div>
						<?php wp_nonce_field( $this->basename, 'cptch_nonce_name' ); ?>
						<input type="hidden" name="cptch_save_add_ip_form" value="1" />
					<?php } ?>
				</form>
			<?php } ?>
			<form class="form-table cptch_allowlist_form" method="post" action="admin.php?page=captcha-allowlist.php" style="margin: 10px 0;<?php echo ! ( isset( $_REQUEST['cptch_show_allowlist_form'] ) || isset( $_REQUEST['cptch_add_to_allowlist'] ) ) ? 'display: none;': ''; ?>">
				<div>
					<label><?php esc_html_e( 'IP to allow list', 'captcha-plus' ); ?></label>
					<br />
					<input type="text" maxlength="31" name="cptch_add_to_allowlist" />
					<?php if ( isset( $my_ip ) ) { ?>
						<br />
						<label id="cptch_add_my_ip">
							<input type="checkbox" name="cptch_add_to_allowlist_my_ip" value="1" />
							<?php esc_html_e( 'My IP', 'captcha-plus' ); ?>
							<input type="hidden" name="cptch_add_to_allowlist_my_ip_value" value="<?php echo $my_ip; ?>" />
						</label>
					<?php } ?>
					<div>
						<span class="bws_info" style="line-height: 2;"><?php esc_html_e( "Allowed formats", 'captcha-plus' ); ?>:&nbsp;<code>192.168.0.1</code></span>
						<br/>
						<span class="bws_info" style="line-height: 2;"><?php esc_html_e( "Allowed diapason", 'captcha-plus' ); ?>:&nbsp;<code>0.0.0.0 - 255.255.255.255</code></span>
					</div>
					<p>
						<input type="submit" id="cptch_add_to_allowlist_button" class="button-secondary" value="<?php esc_html_e( 'Add IP to allow list', 'captcha-plus' ) ?>" />
						<?php wp_nonce_field( $this->basename, 'cptch_nonce_name' ); ?>
					</p>
				</div>
			</form>
			<?php if ( ! $this->disable_list) { ?>
				<form id="cptch_allowlist_search" method="post" action="admin.php?page=captcha-allowlist.php">
					<?php $this->search_box( esc_html__( 'Search item', 'captcha-plus' ), 'search_allowlisted_ip' );
					wp_nonce_field( $this->basename, 'cptch_nonce_name' ); ?>
				</form>
				<form id="cptch_allowlist" method="post" action="admin.php?page=captcha-allowlist.php">
					<?php $this->display();
					wp_nonce_field( $this->basename, 'cptch_nonce_name' ); ?>
				</form>
			<?php }
		}

		/**
		* Function to prepare data before display
		* @return void
		*/
		function prepare_items() {

			if ( isset( $_GET['orderby'] ) && in_array( $_GET['orderby'], array_keys( $this->get_sortable_columns() ) ) ) {
				switch ( $_GET['orderby'] ) {
					case 'ip':
						$this->order_by = 'ip_from_int';
						break;
					case 'ip_from':
						$this->order_by = 'ip_from_int';
						break;
					case 'ip_to':
						$this->order_by = 'ip_to_int';
						break;
					default:
						$this->order_by = esc_sql( $_GET['orderby'] );
						break;
				}
			} else {
				$this->order_by = 'add_time';
			}
			$this->order		= isset( $_REQUEST['order'] ) && in_array( strtoupper( $_REQUEST['order'] ), array( 'ASC', 'DESC' ) ) ? $_REQUEST['order'] : '';
			$this->paged		= isset( $_REQUEST['paged'] ) && is_numeric( $_REQUEST['paged'] ) ? $_REQUEST['paged'] : '';
			$this->s			= isset( $_REQUEST['s'] ) ? esc_html( trim( $_REQUEST['s'] ) ) : '';
			$this->per_page		= $this->get_items_per_page( 'cptch_per_page', 20 );

			$columns				= $this->get_columns();
			$hidden					= array();
			$sortable				= $this->get_sortable_columns();
			$this->_column_headers	= array( $columns, $hidden, $sortable );
			$this->items			= $this->get_content();
			$current_page			= $this->get_pagenum();
			$this->set_pagination_args( array(
					'total_items'	=> $this->get_items_number(),
					'per_page'		=> 20,
				)
			);
		}
		/**
		* Function to show message if empty list
		* @return void
		*/
		function no_items() {
			$label = isset( $_REQUEST['s'] ) ? esc_html__( 'Nothing found', 'captcha-plus' ) : esc_html__( 'No IP in the allow list', 'captcha-plus' ); ?>
			<p><?php echo $label; ?></p>
		<?php }

		function get_columns() {
			$columns = array(
				'cb'		=> '<input type="checkbox" />',
				'ip'		=> esc_html__( 'IP Address', 'captcha-plus' ),
				'add_time'	=> esc_html__( 'Date Added', 'captcha-plus' )
			);
			return $columns;
		}
		/**
		 * Get a list of sortable columns.
		 * @return array list of sortable columns
		 */
		function get_sortable_columns() {
			$sortable_columns = array(
				'ip'		=> array( 'ip', true ),
				'add_time'	=> array( 'add_time', false )
			);
			return $sortable_columns;
		}
		/**
		 * Fires when the default column output is displayed for a single row.
		 * @param      string    $column_name      The custom column's name.
		 * @param      array     $item             The cuurrent letter data.
		 * @return    void
		 */
		function column_default( $item, $column_name ) {
			switch ( $column_name ) {
				case 'ip':
				case 'add_time':
					return $item[ $column_name ];
				default:
					/* Show whole array for bugfix */
					return print_r( $item, true );
			}
		}
		/**
		 * Function to manafe content of column with checboxes
		 * @param     array     $item        The cuurrent letter data.
		 * @return    string                  with html-structure of <input type=['checkbox']>
		 */
		function column_cb( $item ) {
			/* customize displaying cb collumn */
			return sprintf(
				'<input type="checkbox" name="id[]" value="%s" />', $item['id']
			);
		}
		/**
		 * Function to manafe content of column with IP-adresses
		 * @param     array     $item        The cuurrent letter data.
		 * @return    string                  with html-structure of <input type=['checkbox']>
		 */
		function column_ip( $item ) {
			$order_by	= empty( $this->order_by ) ? '' : "&orderby={$this->order_by}";
			$order		= empty( $this->order )    ? '' : "&order={$this->order}";
			$paged		= empty( $this->paged )    ? '' : "&paged={$this->paged}";
			$s			= empty( $this->s )        ? '' : "&s={$this->s}";
			$url		= "?page=captcha-allowlist.php&cptch_remove={$item['id']}{$order_by}{$order}{$paged}{$s}";
			$actions	= array(
				'delete' => '<a href="' . wp_nonce_url( $url, "cptch_nonce_remove_{$item['id']}" ) . '">' . esc_html__( 'Delete', 'captcha-plus' ) . '</a>'
			);
			return sprintf('%1$s %2$s', $item['ip'], $this->row_actions( $actions ) );
		}
		/**
		 * List with bulk action for IP
		 * @return array   $actions
		 */
		function get_bulk_actions() {
			return $this->disable_list ? array() : array( 'cptch_remove'=> esc_html__( 'Delete', 'captcha-plus' ) );
		}
		/**
		 * Get content for table
		 * @return  array
		 */
		function get_content() {
			global $wpdb;

			if ( empty( $this->s ) ) {
				$where = '';
			} else {
				$ip_int = filter_var( $this->s, FILTER_VALIDATE_IP ) ? sprintf( '%u', ip2long( $this->s ) ) : 0;
				$where =
						0 == $ip_int
					?
						" WHERE `ip` LIKE '%{$this->s}%'"
					:
						" WHERE ( `ip_from_int` <= {$ip_int} AND `ip_to_int` >= {$ip_int} )";
			}

			/**
			* filter items by date if it is needed
			*/
			if ( 'all' != $this->filter_by ) {
				$now = time();
				$day = 60 * 60 * 24;

				/**
				* get the quantity of seconds in the day
				*/
				switch ( $this->filter_by ) {
					case 'day':
						$point = $now - $day;
						break;
					case 'week':
						$point = $now - $day * 7;
						break;
					case 'month':
						$point = $now - $day * 31;
						break;
					case 'year':
						$point = $now - $day * 365;
						break;
					default:
						break;
				}

				if( ! empty( $point ) ) {
					$point = date( 'Y-m-d h:i:s', $point );

					$where .= ! empty( $where ) ? ' &&' : 'WHERE';
					$where .= ' `add_time` > "' . $point . '"';
				}
			}

			$order_by	= empty( $this->order_by ) ? '' : " ORDER BY `{$this->order_by}`";
			$order		= empty( $this->order )    ? '' : strtoupper( " {$this->order}" );
			$offset		= empty( $this->paged )    ? '' : " OFFSET " . ( $this->per_page * ( absint( $this->paged ) - 1 ) );

			return $wpdb->get_results( "SELECT * FROM `{$wpdb->prefix}cptch_allowlist`{$where}{$order_by}{$order} LIMIT {$this->per_page}{$offset}", ARRAY_A );
		}

		/**
		 * Get number of all IPs which were added to database
		 * @since  1.1.4
		 * @param  void
		 * @return int    the number of IPs
		 */
		private function get_items_number() {
			global $wpdb;
			if ( empty( $this->s ) ) {
				$where = '';
			} else {
				$ip_int = filter_var( $this->s, FILTER_VALIDATE_IP ) ? sprintf( '%u', ip2long( $this->s ) ) : 0;
				$where =
						0 == $ip_int
					?
						" WHERE `ip` LIKE '%{$this->s}%'"
					:
						" WHERE ( `ip_from_int` <= {$ip_int} AND `ip_to_int` >= {$ip_int} )";
			}
			return absint( $wpdb->get_var( "SELECT COUNT(`id`) FROM `{$wpdb->prefix}cptch_allowlist`{$where}" ) );
		}

		/**
		 * This function display's top- & bottom- filters
		 * @since 4.3.1
		 * @param string $which
		 * @return void
		 */
		function extra_tablenav( $which ) { ?>
			<select name="cptch_date_filter[]">
				<?php foreach ( $this->date_filter_options as $key => $value) { ?>
					<option value="<?php echo $key;?>" <?php echo ( $key == $this->filter_by ) ? 'selected' : ''; ?>>
						<?php echo $value; ?>
					</option>
				<?php } ?>

			</select>

			<input type="hidden" name="cptch_last_filtred_by" value="<?php echo $this->filter_by; ?>" />
			<input type="submit" class="button action" value="<?php esc_html_e( 'Filter', 'captcha-plus' ); ?>" />
		<?php }

		/**
		 * Check if filtring option is valid and not same to last time used filtring option
		 * @see $this->extra_tablenav
		 * @param  string   filtring option
		 * @return array    filtred $_POST['cptch_date_filter']
		 */
		private function get_date_filter_values( $item ) {
			return ( in_array( $item, array_keys( $this->date_filter_options ) ) && $this->last_filtred_by != $item );
		}

		/**
		 * Handle necessary reqquests and display notices
		 * @return void
		 */
		function display_notices() {
			global $wpdb, $cptch_options;
			$error = $message = '';

			if ( isset( $_POST['bws_hide_premium_options'] ) ) {
				return;
			}

			$bulk_action = isset( $_REQUEST['action'] ) && 'cptch_remove' == $_REQUEST['action'] ? true : false;
			if ( ! $bulk_action ) {
				$bulk_action = isset( $_REQUEST['action2'] ) && 'cptch_remove' == $_REQUEST['action2'] ? true : false;
			}

			/* Add IP in to database */
			if ( isset( $_POST['cptch_add_to_allowlist'] ) && ( ! empty( $_POST['cptch_add_to_allowlist'] ) || isset( $_POST['cptch_add_to_allowlist_my_ip'] ) ) && check_admin_referer( $this->basename, 'cptch_nonce_name' ) ) {
				$add_ip = isset( $_POST['cptch_add_to_allowlist_my_ip'] ) ? $_POST['cptch_add_to_allowlist_my_ip_value'] : $_POST['cptch_add_to_allowlist'];

				$valid_ip = filter_var( stripslashes( esc_html( trim( $add_ip ) ) ), FILTER_VALIDATE_IP );
				if ( $valid_ip ) {
					$ip_int = sprintf( '%u', ip2long( $valid_ip ) );
					$id = $wpdb->get_var( "SELECT `id` FROM " . $wpdb->prefix . "cptch_allowlist WHERE ( `ip_from_int` <= " . $ip_int . " AND `ip_to_int` >= " . $ip_int . " ) OR `ip` LIKE '" . $valid_ip . "' LIMIT 1;" );
					/* check if IP already in database */
					if ( is_null( $id ) ) {
						$time = date( 'Y-m-d H:i:s', current_time( 'timestamp' ) );
						$wpdb->insert(
							$wpdb->prefix . "cptch_allowlist",
							array(
								'ip'			=> $valid_ip,
								'ip_from_int'	=> $ip_int,
								'ip_to_int'		=> $ip_int,
								'add_time'		=> $time
							)
						);
						if ( ! $wpdb->last_error ) {
							$message = esc_html__( 'IP added to the allow list successfully.', 'captcha-plus' );
						} else {
							$error = esc_html__( 'Some errors occurred.', 'captcha-plus' );
						}
					} else {
						$error = esc_html__( 'IP is already in the allow list.', 'captcha-plus' );
					}
				} else {
					$error = esc_html__( 'Invalid IP. See allowed formats.', 'captcha-plus' );
				}
				if ( empty( $error ) ) {
					$cptch_options['allowlist_is_empty'] = false;
					update_option( 'cptch_options', $cptch_options );
				}
			} elseif ( $bulk_action && check_admin_referer( $this->basename, 'cptch_nonce_name' ) ) {
				if ( ! empty( $_REQUEST['id'] ) ) {
					$list   = implode( ',', $_REQUEST['id'] );
					$result	= $wpdb->query( $wpdb->prepare( "DELETE FROM `" . $wpdb->prefix . "cptch_allowlist` WHERE `id` IN (%s);", $list ) );
					if ( ! $wpdb->last_error ) {
						$message = sprintf( _n( "%s IP was deleted successfully", "%s IPs were deleted successfully", $result, 'captcha-plus' ), $result );
						if ( ! is_null( $wpdb->get_var( "SELECT `id` FROM `{$wpdb->prefix}cptch_allowlist` LIMIT 1" ) ) ) {
							$cptch_options['allowlist_is_empty'] = false;
							update_option( 'cptch_options', $cptch_options );
						}
					} else {
						$error = esc_html__( 'Some errors occurred', 'captcha-plus' );
					}
				}
			} elseif ( isset( $_GET['cptch_remove'] ) && check_admin_referer( 'cptch_nonce_remove_' . $_GET['cptch_remove'] ) ) {
				$wpdb->delete( $wpdb->prefix . "cptch_allowlist", array( 'id' => $_GET['cptch_remove'] ) );
				if ( ! $wpdb->last_error ) {
					$message = esc_html__( "One IP was deleted successfully", 'captcha-plus' );
					if( ! is_null( $wpdb->get_var( "SELECT `id` FROM `{$wpdb->prefix}cptch_allowlist` LIMIT 1" ) ) ) {
						$cptch_options['allowlist_is_empty'] = false;
						update_option( 'cptch_options', $cptch_options );
					}
				} else {
					$error = esc_html__( 'Some errors occurred', 'captcha-plus' );
				}
			} elseif ( isset( $_POST['cptch_add_to_allowlist'] ) && empty( $_POST['cptch_add_to_allowlist'] ) ) {
				$error = esc_html__( 'You have not entered any value.', 'captcha-plus' );
			} elseif ( isset( $_REQUEST['s'] ) ) {
				if ( '' == $_REQUEST['s'] ) {
					$error = esc_html__( 'You have not entered any value in to the search form.', 'captcha-plus' );
				} else {
					$message = esc_html__( 'Search results for', 'captcha-plus' ) . '&nbsp;:&nbsp;' . esc_html( $_REQUEST['s'] );
				}
			} elseif ( isset( $_POST['cptch_load_limit_attempts_allowlist'] ) && check_admin_referer( $this->basename, 'cptch_nonce_name' ) ) {
				/* copy data from the allow list of LimitAttempts plugin */
				$time = date( 'Y-m-d H:i:s', current_time( 'timestamp' ) );
				$column_exists = $wpdb->query( "SHOW COLUMNS FROM `" . $wpdb->prefix . "lmtttmpts_allowlist` LIKE 'ip_from_int'" );
				/* LimitAttempts Free hasn't  `ip_from_int`, `ip_to_int` COLUMNS */
				if ( 0 == $column_exists ) {
					$result = 0;
					$all_ip = $wpdb->get_results( "SELECT `ip`, '{$time}' FROM `{$wpdb->prefix}lmtttmpts_allowlist`", ARRAY_A );
					foreach ( $all_ip as $ip_value ) {
						$ip_int = sprintf( '%u', ip2long( $ip_value['ip'] ) );
						$result_single = $wpdb->query( $wpdb->prepare(
							"INSERT IGNORE INTO `{$wpdb->prefix}cptch_allowlist`
							( `ip`, `ip_from_int`, `ip_to_int`, `add_time` ) VALUES ( %s, %s, %s, %s );",
							$ip_value['ip'], $ip_int, $ip_int, $time ) );
						$result = $result + $result_single;
					}
				} else {
					$result = $wpdb->query(
						"INSERT IGNORE INTO `{$wpdb->prefix}cptch_allowlist`
							( `ip`, `ip_from_int`, `ip_to_int`, `add_time` )
							( SELECT `ip`, `ip_from_int`, `ip_to_int`, '{$time}'
								FROM `{$wpdb->prefix}lmtttmpts_allowlist` );"
					);
				}

				if ( $wpdb->last_error ) {
					$error = $wpdb->last_error;
				} else {
					$message = $result . '&nbsp;' . esc_html__( 'IP-address(es) successfully copied to the allow list.', 'captcha-plus' );
					$cptch_options['allowlist_is_empty'] = false;
					update_option( 'cptch_options', $cptch_options );
				}
			}
			if ( ! empty( $message ) ) { ?>
				<div class="updated fade inline"><p><strong><?php echo $message; ?></strong></p></div>
			<?php }
			if ( ! empty( $error ) ) { ?>
				<div class="error inline"><p><strong><?php echo $error; ?></strong></p></div>
			<?php }
		}

		/*
		 * Get info about plugins Limit Attempts ( Free or Pro ) by BestWebSoft
		 */
		function get_limit_attempts_info() {
			global $wp_version, $cptch_plugin_info;

			if ( 'active' == $this->la_info['status'] ) {
				$data = array(
					'active'			=> true,
					'name'				=> $this->la_info['plugin_info']["Name"],
					'label'				=> esc_html__( 'use', 'captcha-plus' ) . '&nbsp;<a href="?page=' . $this->la_info['plugin_info']["TextDomain"] . '.php">' . esc_html__( 'the allow list of', 'captcha-plus' ) . '&nbsp;' . $this->la_info['plugin_info']["Name"] . '</a>',
					'notice'			=> '',
					'disabled'			=> false,
				);
			} elseif ( 'deactivated' == $this->la_info['status'] ) {
				$data = array(
					'active'			=> false,
					'name'				=> $this->la_info['plugin_info']["Name"],
					'label'				=> sprintf( esc_html__( 'use the allow list of %s', 'captcha-plus' ), $this->la_info['plugin_info']["Name"] ),
					'notice'			=> sprintf( esc_html__( 'you should %s to use this functionality', 'captcha-plus' ), '<a href="plugins.php">' . esc_html__( 'activate', 'captcha-plus' ) . '&nbsp;' . $this->la_info['plugin_info']["Name"] . '</a>' ),
					'disabled'			=> true,
					'link'				=> '<a href="plugins.php">' . esc_html__( 'Activate', 'captcha-plus' ) . '</a>'
				);
			} elseif ( 'not_installed' == $this->la_info['status'] ) {
				$data = array(
					'active'			=> false,
					'name'				=> 'Limit Attempts by BestWebSoft',
					'label'				=> sprintf( esc_html__( 'use the allow list of %s', 'captcha-plus' ), 'Limit Attempts by BestWebSoft' ),
					'notice'			=> sprintf( esc_html__( 'you should install %s to use this functionality', 'captcha-plus' ), '<a href="https://bestwebsoft.com/products/wordpress/plugins/limit-attempts?k=9ab9d358ad3a23b8a99a8328595ede2e&pn=72&v=' . $cptch_plugin_info["Version"] . '&wp_v=' . $wp_version . '" target="_blank">Limit Attempts by BestWebSoft</a>' ),
					'disabled'			=> true,
					'link'				=> '<a href="https://bestwebsoft.com/products/wordpress/plugins/limit-attempts?k=9ab9d358ad3a23b8a99a8328595ede2e&pn=72&v=' . $cptch_plugin_info["Version"] . '&wp_v=' . $wp_version . '" target="_blank">' . esc_html__( 'Install Now', 'captcha-plus' ) . '</a>'
				);
			}
			return $data;
		}
	}
}