<?php

/**
 * Register all actions and filters for the plugin
 *
 * @link       http://echopointer.com
 * @since      2.0.0
 *
 * @package    Kas_Dvf
 * @subpackage Kas_Dvf/classes
 */

/**
 * Register all actions and filters for the plugin.
 *
 * Maintain a list of all hooks that are registered throughout
 * the plugin, and register them with the WordPress API. Call the
 * run function to execute the list of actions and filters.
 *
 * @package    Kas_Dvf
 * @subpackage Kas_Dvf/classes
 * @author     WeDevs
 * @utilized   Syed Muhammad Shafiq <shafiq_shaheen@hotmail.com>
 */
class Kas_Dvf_Query {
    
    /**
     * The ID of this plugin.
     *
     * @since    2.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;
    
    /**
     * The version of this plugin.
     *
     * @since    2.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;
    
    /**
     * Holds the WP_User_Query object
     *
     * @since 2.0.0
     *
     * @var null|WP_User_Query
     */
    private $user_query = null;
    
    /**
     * Latitude query value
     *
     * @since 2.0.0
     *
     * @var float
     */
    private $latitude = 0.00;
    
    /**
     * Longitude query value
     *
     * @since 2.0.0
     *
     * @var float
     */
    private $longitude = 0.00;
    
    /**
     * Distance/Radius query value
     *
     * @since 2.0.0
     *
     * @var int
     */
    private $distance = 0;
    
    /**
     * Class constructor
     *
     * @since 2.0.0
     *
     * @return void
     */
    public function __construct( $plugin_name, $version ) {
        
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        

    }
    
    /**
     * Add action to filter vendor sql query
     *
     * @since 2.0.0
     *
     * @param array $seller_args
     *
     * @return array
     */
    public function filter_pre_user_query( $seller_args ) {
        add_action( 'pre_user_query', array( $this, 'add_user_query' ) );
        
        return $seller_args;
    }
    
    /**
     * WP_User_Query hook
     *
     * @since 2.0.0
     *
     * @param WP_User_Query $user_query
     */
    public function add_user_query( $user_query ) {
        
        if ( version_compare( DOKAN_PLUGIN_VERSION, '2.9.6', '>' ) ) {
            if ( empty( $user_query->query_vars['role__in'] ) || $user_query->query_vars['role__in'] !== ['seller', 'administrator'] ) {
                return;
            }
        } else {
            if ( empty( $user_query->query_vars['role'] ) || 'seller' !== $user_query->query_vars['role'] ) {
                return;
            }
        }
        $s_options = get_option($this->plugin_name . '-scripts');
        $this->user_query = $user_query;
        
        $this->latitude  = isset( $_SESSION['current_latitude'] ) ? $_SESSION['current_latitude'] : $this->latitude;
        $this->longitude = isset( $_SESSION['current_longitude'] ) ? $_SESSION['current_longitude'] : $this->longitude;
        $this->distance  = isset( $s_options['kas-search-radius'] ) ? $s_options['kas-search-radius'] : 0;
        
        $this->filter_query_fields();
        $this->filter_query_from();
        $this->filter_query_orderby();
        
    }
    
    /**
     * Add extra select statement to WP_User_Query
     *
     * @since 2.0.0
     *
     * @return void
     */
    private function filter_query_fields() {
        $g_options = get_option($this->plugin_name . '-general');
        $this->user_query->query_fields .= ', metalat.meta_value as dokan_geo_latitude, metalong.meta_value as dokan_geo_longitude, metaaddr.meta_value as dokan_geo_address';
        
        if ( $this->latitude && $this->longitude ) {
            // unit in kilometers or miles
            $distance_unit = (isset($g_options['kas-distance-type']) ? $g_options['kas-distance-type'] : 'imperial');
            
            $distance_earth_center_to_surface = ( 'metric' === $distance_unit ) ? 6371 : 3959;
            
            $this->user_query->query_fields .= ", (
                {$distance_earth_center_to_surface} * acos(
                    cos( radians( {$this->latitude} ) ) *
                    cos( radians( metalat.meta_value ) ) *
                    cos(
                        radians( metalong.meta_value ) - radians( {$this->longitude} )
                    ) +
                    sin( radians( {$this->latitude} ) ) *
                    sin( radians( metalat.meta_value ) )
                )
            ) as geo_distance";
        }
    }
    
    /**
     * Add inner join clauses to WP_User_Query
     *
     * @since 2.0.0
     *
     * @return void
     */
    private function filter_query_from() {
        global $wpdb;
        
        $this->user_query->query_from .= " inner join {$wpdb->usermeta} as metalat on {$wpdb->users}.ID = metalat.user_id and metalat.meta_key = 'dokan_geo_latitude'";
        $this->user_query->query_from .= " inner join {$wpdb->usermeta} as metalong on {$wpdb->users}.ID = metalong.user_id and metalong.meta_key = 'dokan_geo_longitude'";
        $this->user_query->query_from .= " inner join {$wpdb->usermeta} as metaaddr on {$wpdb->users}.ID = metaaddr.user_id and metaaddr.meta_key = 'dokan_geo_address'";
    }
    
    /**
     * Add having clause to WP_User_Query
     *
     * @since 2.0.0
     *
     * @return void
     */
    private function filter_query_orderby() {
        if ( $this->latitude && $this->longitude && $this->distance ) {
            $distance = absint( $this->distance );
            $this->user_query->query_orderby = "having ( geo_distance < {$distance} or geo_distance is null ) " . $this->user_query->query_orderby;
        }
    }
  
}