<?php

namespace LivemeshAddons;

// Exit if accessed directly
use Elementor\Controls_Manager;
use Elementor\Controls_Stack;

if (!defined('ABSPATH'))
    exit;


if (!class_exists('LELA_Code_Generator')) :

    /**
     * Main LELA_Code_Generator Class
     *
     */
    class LELA_Code_Generator {

        /**
         * LELA OpenAI Client Object
         *
         * @var object|LELA_OpenAI_Client
         */
        public $openai_client;

        /**
         * Get things going
         *
         * @since 1.4.4
         */
        public function __construct() {

            add_action('elementor/element/code-highlight/section_content/before_section_end', array($this, 'register_code_generation_controls'), 20, 2);

            add_action('elementor/element/code-highlight/section_content/after_section_end', array($this, 'register_openai_settings_controls'), 20, 2);

            $this->openai_client = new LELA_OpenAI_Client();

            add_action('wp_ajax_nopriv_lela_generate_code', array($this, 'generate_code'));

            add_action('wp_ajax_lela_generate_code', array($this, 'generate_code'));

        }

        /**
         * Register Code Generation Controls. Make sure you prefix everything to avoid conflict with Elementor widget
         *
         * @param Controls_Stack $element Elementor element.
         * @param string $section_id Section ID.
         */
        public function register_code_generation_controls(Controls_Stack $element, $section_id) {

            $element->add_control(
                'lela_code_generation',
                [
                    'label' => esc_html__('AI Code Generation', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before'
                ]
            );

            $element->add_control(
                'lela_prompt',
                [
                    'type' => Controls_Manager::TEXTAREA,
                    'label' => esc_html__('Prompt for Code Generation', 'livemesh-el-assistant'),
                    'label_block' => true,
                    'rows' => 10,
                    'description' => wp_kses_post(__('Provide detailed instructions for the source code that you want to generate <strong>along with the programming language</strong>. You can also paste here existing code and request AI to rewrite that or fix errors for you.', 'livemesh-el-assistant')),
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $element->add_control(
                'lela_generate',
                [
                    'type' => Controls_Manager::BUTTON,
                    'label' => '',
                    'separator' => 'before',
                    'show_label' => false,
                    'text' => esc_html__('Generate', 'livemesh-el-assistant'),
                    'description' => wp_kses_post(__('Once source code is generated, for proper code highlighting, make sure the right programming language is chosen in the <strong>Language</strong> dropdown above the code editor', 'livemesh-el-assistant')),
                    'button_type' => 'default',
                    'event' => 'lela:code:generate'

                ]
            );

        }

        /**
         * Register OpenAI configuration controls. Make sure you prefix everything to avoid conflict with Elementor widget
         *
         * @param Controls_Stack $element Elementor element.
         * @param string $section_id Section ID.
         */
        public function register_openai_settings_controls(Controls_Stack $element, $section_id) {

            $element->start_controls_section(
                'lela_section_openai_api_settings',
                [
                    'label' => esc_html__('OpenAI API Settings', 'livemesh-el-assistant'),
                    'tab' => Controls_Manager::TAB_SETTINGS,
                ]
            );

            $element->add_control(
                'lela_codex_model',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Choose the Codex or GPT Model to use', 'livemesh-el-assistant'),
                    'description' => wp_kses_post(__('Know more about these models and their capabilities <a target="_blank" href="https://platform.openai.com/docs/models/overview">here</a>', 'livemesh-el-assistant')),
                    'options' => array(
                        'text-davinci-003' => esc_html__('text-davinci-003', 'livemesh-el-assistant'),
                        'gpt-3.5-turbo' => esc_html__('gpt-3.5-turbo', 'livemesh-el-assistant'),
                        'gpt-4' => esc_html__('gpt-4 (Limited Access)', 'livemesh-el-assistant'),
                        'gpt-4-32k' => esc_html__('gpt-4-32k (Limited Access)', 'livemesh-el-assistant'),
                        'code-davinci-002' => esc_html__('code-davinci-002 (Deprecated)', 'livemesh-el-assistant'),
                        'code-cushman-002' => esc_html__('code-cushman-002 (Deprecated)', 'livemesh-el-assistant'),
                        'text-curie-001' => esc_html__('text-curie-001', 'livemesh-el-assistant'),
                        'text-babbage-001' => esc_html__('text-babbage-001', 'livemesh-el-assistant'),
                        'text-ada-001' => esc_html__('text-ada-001', 'livemesh-el-assistant'),
                    ),
                    'default' => 'text-davinci-003',
                ]
            );

            $element->add_control(
                'lela_max_tokens',
                [
                    'label' => esc_html__('Maximum Tokens', 'livemesh-el-assistant'),
                    'description' => esc_html__('The maximum number of tokens to use for generation. Higher values means more content will be generated.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 1800,
                    ],
                    'range' => [
                        'px' => [
                            'min' => 1,
                            'max' => 4000,
                        ],
                    ],
                ]
            );

            $element->add_control(
                'lela_temperature',
                [
                    'label' => esc_html__('Temperature', 'livemesh-el-assistant'),
                    'description' => esc_html__('The sampling temperature to use. Higher values means the model will take more risks. Try 0.9 for more creative applications, and 0 for ones with a well-defined answer.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 0.6,
                    ],
                    'range' => [
                        'px' => [
                            'min' => 0,
                            'step' => 0.1,
                            'max' => 1,
                        ],
                    ],
                ]
            );

            $element->add_control(
                'lela_presence_penalty',
                [
                    'label' => esc_html__('Presence Penalty', 'livemesh-el-assistant'),
                    'description' => esc_html__('Number between -2.0 and 2.0. Default is 0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model\'s likelihood to talk about new topics.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 0,
                    ],
                    'range' => [
                        'px' => [
                            'min' => -2.0,
                            'step' => 0.1,
                            'max' => 2.0,
                        ],
                    ],
                ]
            );

            $element->add_control(
                'lela_frequency_penalty',
                [
                    'label' => esc_html__('Frequency Penalty', 'livemesh-el-assistant'),
                    'description' => esc_html__('Number between -2.0 and 2.0. Default is 0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model\'s likelihood to repeat the same line verbatim.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 0,
                    ],
                    'range' => [
                        'px' => [
                            'min' => -2.0,
                            'step' => 0.1,
                            'max' => 2.0,
                        ],
                    ],
                ]
            );

            $element->end_controls_section();

        }

        public function generate_code() {

            check_ajax_referer('lela-assistant', '_ajax_nonce-lela-assistant');

            if (!current_user_can('manage_options')) {
                wp_send_json_error(esc_html__('Error! Your account does not have user permission to generate code!', 'livemesh-el-assistant'));
            }

            $model_params = $code_params = array();

            if (isset($_POST['modelParams'])) {
                $model_params = array_map('sanitize_text_field', $_POST['modelParams']);
            }
            if (isset($_POST['codeParams'])) {
                $code_params = array_map('sanitize_text_field', $_POST['codeParams']);
            }

            $output = '';

            try {

                if (empty($code_params['prompt'])) {
                    throw new \Exception(esc_html__('You have not provided the topic for the source code that you want to generate!', 'livemesh-el-assistant'));
                }

                $prompt = 'Write me source code in the programming language specified as per instructions that follow: ' . $code_params['prompt'] . '\n';;

                $call_params = array(
                    'model' => $model_params['codexModel'],
                    'max_tokens' => intval($model_params['maxTokens']),
                    'temperature' => floatval($model_params['temperature']),
                    'top_p' => 1,
                    'n' => 1,
                    'stream' => false,
                    'frequency_penalty' => floatval($model_params['frequencyPenalty']),
                    'presence_penalty' => floatval($model_params['presencePenalty']),
                );

                $chat_mode = false;
                if($call_params['model'] == 'gpt-3.5-turbo' || $call_params['model'] == 'gpt-4' || $call_params['model'] == 'gpt-4-32k') {
                    $chat_mode = true;
                }

                if ($chat_mode) {
                    $url = 'https://api.openai.com/v1/chat/completions';
                    $call_params['messages'] = array(
                        array('role' => 'user', 'content' => $prompt)
                    );
                }
                else {
                    $url = 'https://api.openai.com/v1/completions';
                    $call_params['prompt'] = $prompt;
                    $call_params['best_of'] = 1;
                }

                $response = $this->openai_client->dispatch($url, $call_params);

                $choices = $response['choices'];

                if ( count( $choices ) == 0 ) {
                    throw new \Exception(esc_html__('No response was generated. Please try again using different prompt!', 'livemesh-el-assistant'));
                }

                if ($chat_mode) {
                    $output = $choices[0]['message']['content'];
                }
                else {
                    $output =$choices[0]['text'];
                }

                wp_send_json_success(trim($output));

            } catch (\Throwable $throwable) {
                wp_send_json_error(esc_html__('Error! ', 'livemesh-el-assistant') . $throwable->getMessage());
            }

        }

    }

endif; // End if class_exists check


