<?php

namespace LivemeshAddons;

// Exit if accessed directly
use Elementor\Controls_Manager;
use Elementor\Controls_Stack;

if (!defined('ABSPATH'))
    exit;


if (!class_exists('LELA_Content_Generator')) :

    /**
     * Main LELA_Content_Generator Class
     *
     */
    class LELA_Content_Generator {

        /**
         * LELA OpenAI Client Object
         *
         * @var object|LELA_OpenAI_Client
         */
        public $openai_client;

        /**
         * Get things going
         *
         * @since 1.4.4
         */
        public function __construct() {

            add_action('elementor/element/text-editor/section_editor/before_section_end', array($this, 'register_content_generation_controls'), 20, 2);

            add_action('elementor/element/text-editor/section_editor/after_section_end', array($this, 'register_content_settings_controls'), 20, 2);

            // Add this last
            add_action('elementor/element/text-editor/section_editor/after_section_end', array($this, 'register_openai_settings_controls'), 30, 2);

            $this->openai_client = new LELA_OpenAI_Client();
            
            add_action('wp_ajax_nopriv_lela_generate_content', array($this, 'generate_content'));

            add_action('wp_ajax_lela_generate_content', array($this, 'generate_content'));
        }

        /**
         * Register Content Generation Controls. Make sure you prefix everything to avoid conflict with Elementor widget
         *
         * @param Controls_Stack $element Elementor element.
         * @param string $section_id Section ID.
         */
        public function register_content_generation_controls(Controls_Stack $element, $section_id) {

            $element->add_control(
                'lela_content_generation',
                [
                    'label' => esc_html__('AI Content Generation', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before'
                ]
            );

            $element->add_control(
                'lela_prompt',
                [
                    'type' => Controls_Manager::TEXTAREA,
                    'label' => esc_html__('Prompt for Content Generation', 'livemesh-el-assistant'),
                    'label_block' => true,
                    'rows' => 10,
                    'description' => esc_html__('Provide subject and detailed instructions for the content you want to generate. You can also paste here existing content and request AI to fix style/grammar or rewrite that content for you', 'livemesh-el-assistant'),
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $element->add_control(
                'lela_keywords',
                [
                    'type' => Controls_Manager::TEXT,
                    'label' => esc_html__('Keywords', 'livemesh-el-assistant'),
                    'label_block' => true,
                    'description' => esc_html__('Provide keywords separated by commas for SEO', 'livemesh-el-assistant'),
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $element->add_control(
                'lela_image_prompt',
                [
                    'type' => Controls_Manager::TEXT,
                    'label' => esc_html__('Prompt for Image', 'livemesh-el-assistant'),
                    'label_block' => true,
                    'description' => esc_html__('Specify instructions for image generation. Leave blank if you do not want image to be generated', 'livemesh-el-assistant'),
                    'dynamic' => [
                        'active' => true,
                    ],
                ]
            );

            $element->add_control(
                'lela_generate',
                [
                    'type' => Controls_Manager::BUTTON,
                    'label' => '',
                    'separator' => 'before',
                    'show_label' => false,
                    'text' => esc_html__('Generate', 'livemesh-el-assistant'),
                    'description' => esc_html__('You can control settings for content generation like writing style, writing tone, image style etc. in the Settings tab', 'livemesh-el-assistant'),
                    'button_type' => 'default',
                    'event' => 'lela:content:generate'

                ]
            );

        }

        /**
         * Register Content settings controls. Make sure you prefix everything to avoid conflict with Elementor widget
         *
         * @param Controls_Stack $element Elementor element.
         * @param string $section_id Section ID.
         */
        public function register_content_settings_controls(Controls_Stack $element, $section_id) {

            $element->start_controls_section(
                'lela_section_content_params',
                [
                    'label' => esc_html__('AI Content Parameters', 'livemesh-el-assistant'),
                    'tab' => Controls_Manager::TAB_SETTINGS,
                ]
            );

            $element->add_control(
                'lela_language',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Choose Language', 'livemesh-el-assistant'),
                    'description' => esc_html__('The text will be generated in the selected language.', 'livemesh-el-assistant'),
                    'options' => array(
                        'English' => esc_html__('English', 'livemesh-el-assistant'),
                        'Spanish' => esc_html__('Spanish', 'livemesh-el-assistant'),
                        'German' => esc_html__('German', 'livemesh-el-assistant'),
                        'French' => esc_html__('French', 'livemesh-el-assistant'),
                        'Italian' => esc_html__('Italian', 'livemesh-el-assistant'),
                        'Portuguese' => esc_html__('Portuguese', 'livemesh-el-assistant'),
                        'Romanian' => esc_html__('Romanian', 'livemesh-el-assistant'),
                        'Dutch' => esc_html__('Dutch', 'livemesh-el-assistant'),
                        'Polish' => esc_html__('Polish', 'livemesh-el-assistant'),
                        'Finnish' => esc_html__('Finnish', 'livemesh-el-assistant'),
                        'Danish' => esc_html__('Danish', 'livemesh-el-assistant'),
                        'Norwegian' => esc_html__('Norwegian', 'livemesh-el-assistant'),
                        'Chinese' => esc_html__('Chinese', 'livemesh-el-assistant'),
                        'Hindi' => esc_html__('Hindi', 'livemesh-el-assistant'),
                        'Japanese' => esc_html__('Japanese', 'livemesh-el-assistant'),
                        'Korean' => esc_html__('Korean', 'livemesh-el-assistant'),
                        'Indonesian' => esc_html__('Indonesian', 'livemesh-el-assistant'),
                        'Russian' => esc_html__('Russian', 'livemesh-el-assistant'),
                        'Arabic' => esc_html__('Arabic', 'livemesh-el-assistant'),
                        'Persian' => esc_html__('Persian', 'livemesh-el-assistant'),
                        'Turkish' => esc_html__('Turkish', 'livemesh-el-assistant'),
                        'Ukrainian' => esc_html__('Ukrainian', 'livemesh-el-assistant'),
                        'Swedish' => esc_html__('Swedish', 'livemesh-el-assistant'),
                        'Hungarian' => esc_html__('Hungarian', 'livemesh-el-assistant'),
                        'Greek' => esc_html__('Greek', 'livemesh-el-assistant'),
                    ),
                    'default' => 'English',
                ]
            );

            $element->add_control(
                'lela_writing_style',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Writing Style', 'livemesh-el-assistant'),
                    'description' => esc_html__('Specify the appropriate writing style.', 'livemesh-el-assistant'),
                    'options' => array(
                        'Informative' => esc_html__('Informative', 'livemesh-el-assistant'),
                        'Descriptive' => esc_html__('Descriptive', 'livemesh-el-assistant'),
                        'Creative' => esc_html__('Creative', 'livemesh-el-assistant'),
                        'Blog' => esc_html__('Blog', 'livemesh-el-assistant'),
                        'News' => esc_html__('News', 'livemesh-el-assistant'),
                        'Casual' => esc_html__('Casual', 'livemesh-el-assistant'),
                        'Technical' => esc_html__('Technical', 'livemesh-el-assistant'),
                        'Concise' => esc_html__('Concise', 'livemesh-el-assistant'),
                        'Narrative' => esc_html__('Narrative', 'livemesh-el-assistant'),
                        'Reflective' => esc_html__('Reflective', 'livemesh-el-assistant'),
                        'Academic' => esc_html__('Academic', 'livemesh-el-assistant'),
                        'Analytical' => esc_html__('Analytical', 'livemesh-el-assistant'),
                        'Evaluative' => esc_html__('Evaluative', 'livemesh-el-assistant'),
                        'Journalistic' => esc_html__('Journalistic', 'livemesh-el-assistant'),
                        'Argumentative' => esc_html__('Argumentative', 'livemesh-el-assistant'),
                        'Dramatic' => esc_html__('Dramatic', 'livemesh-el-assistant'),
                        'Dialogue' => esc_html__('Dialogue', 'livemesh-el-assistant'),
                        'Satirical' => esc_html__('Satirical', 'livemesh-el-assistant'),
                        'Critical' => esc_html__('Critical', 'livemesh-el-assistant'),
                        'Persuasive' => esc_html__('Persuasive', 'livemesh-el-assistant'),
                        'Detailed' => esc_html__('Detailed', 'livemesh-el-assistant'),
                        'Monologue' => esc_html__('Monologue', 'livemesh-el-assistant'),
                        'Letter' => esc_html__('Letter', 'livemesh-el-assistant'),
                        'Lyrical' => esc_html__('Lyrical', 'livemesh-el-assistant'),
                        'Poetic' => esc_html__('Poetic', 'livemesh-el-assistant'),
                        'Articulate' => esc_html__('Articulate', 'livemesh-el-assistant'),
                        'Anecdotal' => esc_html__('Anecdotal', 'livemesh-el-assistant'),
                        'Personal' => esc_html__('Personal', 'livemesh-el-assistant'),
                        'Comparative' => esc_html__('Comparative', 'livemesh-el-assistant'),
                        'Fiction' => esc_html__('Fiction', 'livemesh-el-assistant'),
                        'Pastoral' => esc_html__('Pastoral', 'livemesh-el-assistant'),
                        'Sensory' => esc_html__('Sensory', 'livemesh-el-assistant'),
                        'Simple' => esc_html__('Simple', 'livemesh-el-assistant'),
                        'Vivid' => esc_html__('Vivid', 'livemesh-el-assistant'),
                        'Historical' => esc_html__('Historical', 'livemesh-el-assistant'),
                        'Biographical' => esc_html__('Biographical', 'livemesh-el-assistant'),
                        'Colloquial' => esc_html__('Colloquial', 'livemesh-el-assistant'),
                        'Expository' => esc_html__('Expository', 'livemesh-el-assistant'),
                    ),
                    'default' => 'Informative',
                ]
            );

            $element->add_control(
                'lela_writing_tone',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Writing Tone', 'livemesh-el-assistant'),
                    'description' => esc_html__('Specify the tone for the generated content.', 'livemesh-el-assistant'),
                    'options' => array(
                        'Neutral' => esc_html__('Neutral', 'livemesh-el-assistant'),
                        'Formal' => esc_html__('Formal', 'livemesh-el-assistant'),
                        'Informal' => esc_html__('Informal', 'livemesh-el-assistant'),
                        'Conversational' => esc_html__('Conversational', 'livemesh-el-assistant'),
                        'Scientific' => esc_html__('Scientific', 'livemesh-el-assistant'),
                        'Assertive' => esc_html__('Assertive', 'livemesh-el-assistant'),
                        'Cheerful' => esc_html__('Cheerful', 'livemesh-el-assistant'),
                        'Inspirational' => esc_html__('Inspirational', 'livemesh-el-assistant'),
                        'Journalistic' => esc_html__('Journalistic', 'livemesh-el-assistant'),
                        'Emotional' => esc_html__('Emotional', 'livemesh-el-assistant'),
                        'Narrative' => esc_html__('Narrative', 'livemesh-el-assistant'),
                        'Sarcastic' => esc_html__('Sarcastic', 'livemesh-el-assistant'),
                        'Factual' => esc_html__('Factual', 'livemesh-el-assistant'),
                        'Humorous' => esc_html__('Humorous', 'livemesh-el-assistant'),
                        'Friendly' => esc_html__('Friendly', 'livemesh-el-assistant'),
                        'Professional' => esc_html__('Professional', 'livemesh-el-assistant'),
                        'Condescending' => esc_html__('Condescending', 'livemesh-el-assistant'),
                        'Persuasive' => esc_html__('Persuasive', 'livemesh-el-assistant'),
                        'Polite' => esc_html__('Polite', 'livemesh-el-assistant'),
                        'Supportive' => esc_html__('Supportive', 'livemesh-el-assistant'),
                        'Sensitive' => esc_html__('Sensitive', 'livemesh-el-assistant'),
                        'Sincere' => esc_html__('Sincere', 'livemesh-el-assistant'),
                        'Skeptical' => esc_html__('Skeptical', 'livemesh-el-assistant'),
                    ),
                    'default' => 'Neutral',
                ]
            );

            $element->end_controls_section();

            $element->start_controls_section(
                'lela_section_image_params',
                [
                    'label' => esc_html__('AI Image Parameters', 'livemesh-el-assistant'),
                    'tab' => Controls_Manager::TAB_SETTINGS,
                ]
            );

            $element->add_control(
                'lela_image_style',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Choose Image Style', 'livemesh-el-assistant'),
                    'description' => esc_html__('Specify the style for the image generated.', 'livemesh-el-assistant'),
                    'options' => array(
                        'None' => esc_html__('- Do Not Specify -', 'livemesh-el-assistant'),
                        '3d' => esc_html__('3D', 'livemesh-el-assistant'),
                        '60s Flat Illustration' => esc_html__('60s Flat Illustration', 'livemesh-el-assistant'),
                        'Abstract' => esc_html__('Abstract', 'livemesh-el-assistant'),
                        'Modern' => esc_html__('Modern', 'livemesh-el-assistant'),
                        'Cartoon' => esc_html__('Cartoon', 'livemesh-el-assistant'),
                        'Comic Book' => esc_html__('Comic Book', 'livemesh-el-assistant'),
                        'Cyberpunk' => esc_html__('Cyberpunk', 'livemesh-el-assistant'),
                        'Fantasy' => esc_html__('Fantasy', 'livemesh-el-assistant'),
                        'Futurism' => esc_html__('Futurism', 'livemesh-el-assistant'),
                        'Noir' => esc_html__('Noir', 'livemesh-el-assistant'),
                        'Oil Painting' => esc_html__('Oil Painting', 'livemesh-el-assistant'),
                        'Pencil Sketch' => esc_html__('Pencil Sketch', 'livemesh-el-assistant'),
                        'Photorealism' => esc_html__('Photorealism', 'livemesh-el-assistant'),
                        'Stained Glass' => esc_html__('Stained Glass', 'livemesh-el-assistant'),
                        'Synthwave' => esc_html__('Synthwave', 'livemesh-el-assistant'),
                        'Graffiti' => esc_html__('Graffiti', 'livemesh-el-assistant'),
                        'Contemporary' => esc_html__('Contemporary', 'livemesh-el-assistant'),
                        'Surrealism' => esc_html__('Surrealism', 'livemesh-el-assistant'),
                        'Popart' => esc_html__('Popart', 'livemesh-el-assistant'),
                        'Impressionist' => esc_html__('Impressionist', 'livemesh-el-assistant'),
                        'Cubism' => esc_html__('Cubism', 'livemesh-el-assistant'),
                        'Watercolor' => esc_html__('Watercolor', 'livemesh-el-assistant'),
                    ),
                    'default' => 'None',
                ]
            );

            $element->add_control(
                'lela_image_size',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Choose Image Size', 'livemesh-el-assistant'),
                    'description' => esc_html__('Specify the size for the image to be generated.', 'livemesh-el-assistant'),
                    'options' => array(
                        '256x256' => esc_html__('Small (256x256)', 'livemesh-el-assistant'),
                        '512x512' => esc_html__('Medium (512x512)', 'livemesh-el-assistant'),
                        '1024x1024' => esc_html__('Big (1024x1024)', 'livemesh-el-assistant'),
                    ),
                    'default' => '512x512',
                ]
            );

            $element->end_controls_section();

        }

        /**
         * Register OpenAI configuration controls. Make sure you prefix everything to avoid conflict with Elementor widget
         *
         * @param Controls_Stack $element Elementor element.
         * @param string $section_id Section ID.
         */
        public function register_openai_settings_controls(Controls_Stack $element, $section_id) {

            $element->start_controls_section(
                'lela_section_openai_api_settings',
                [
                    'label' => esc_html__('OpenAI API Settings', 'livemesh-el-assistant'),
                    'tab' => Controls_Manager::TAB_SETTINGS,
                ]
            );

            $element->add_control(
                'lela_gpt3_model',
                [
                    'type' => Controls_Manager::SELECT,
                    'label' => esc_html__('Choose a GPT Model to use', 'livemesh-el-assistant'),
                    'description' => wp_kses_post(__('Know more about these models and their capabilities <a target="_blank" href="https://platform.openai.com/docs/models/overview">here</a>', 'livemesh-el-assistant')),
                    'options' => array(
                        'text-davinci-003' => esc_html__('text-davinci-003', 'livemesh-el-assistant'),
                        'gpt-3.5-turbo' => esc_html__('gpt-3.5-turbo', 'livemesh-el-assistant'),
                        'gpt-4' => esc_html__('gpt-4 (Limited Access)', 'livemesh-el-assistant'),
                        'gpt-4-32k' => esc_html__('gpt-4-32k (Limited Access)', 'livemesh-el-assistant'),
                        'text-curie-001' => esc_html__('text-curie-001', 'livemesh-el-assistant'),
                        'text-babbage-001' => esc_html__('text-babbage-001', 'livemesh-el-assistant'),
                        'text-ada-001' => esc_html__('text-ada-001', 'livemesh-el-assistant'),
                    ),
                    'default' => 'text-davinci-003',
                ]
            );

            $element->add_control(
                'lela_max_tokens',
                [
                    'label' => esc_html__('Maximum Tokens', 'livemesh-el-assistant'),
                    'description' => esc_html__('The maximum number of tokens to use for generation. Higher values means more content will be generated.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 1000,
                    ],
                    'range' => [
                        'px' => [
                            'min' => 1,
                            'max' => 4000,
                        ],
                    ],
                ]
            );

            $element->add_control(
                'lela_temperature',
                [
                    'label' => esc_html__('Temperature', 'livemesh-el-assistant'),
                    'description' => esc_html__('The sampling temperature to use. Higher values means the model will take more risks. Try 0.9 for more creative applications, and 0 for ones with a well-defined answer.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 0.6,
                    ],
                    'range' => [
                        'px' => [
                            'min' => 0,
                            'step' => 0.1,
                            'max' => 1,
                        ],
                    ],
                ]
            );

            $element->add_control(
                'lela_presence_penalty',
                [
                    'label' => esc_html__('Presence Penalty', 'livemesh-el-assistant'),
                    'description' => esc_html__('Number between -2.0 and 2.0. Default is 0. Positive values penalize new tokens based on whether they appear in the text so far, increasing the model\'s likelihood to talk about new topics.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 0,
                    ],
                    'range' => [
                        'px' => [
                            'min' => -2.0,
                            'step' => 0.1,
                            'max' => 2.0,
                        ],
                    ],
                ]
            );

            $element->add_control(
                'lela_frequency_penalty',
                [
                    'label' => esc_html__('Frequency Penalty', 'livemesh-el-assistant'),
                    'description' => esc_html__('Number between -2.0 and 2.0. Default is 0. Positive values penalize new tokens based on their existing frequency in the text so far, decreasing the model\'s likelihood to repeat the same line verbatim.', 'livemesh-el-assistant'),
                    'type' => Controls_Manager::SLIDER,
                    'default' => [
                        'size' => 0,
                    ],
                    'range' => [
                        'px' => [
                            'min' => -2.0,
                            'step' => 0.1,
                            'max' => 2.0,
                        ],
                    ],
                ]
            );

            $element->end_controls_section();

        }

        public function generate_content() {

            check_ajax_referer('lela-assistant', '_ajax_nonce-lela-assistant');

            if (!current_user_can('manage_options')) {
                wp_send_json_error(esc_html__('Error! Your account does not have user permission to generate content!', 'livemesh-el-assistant'));
            }

            $model_params = $content_params = array();

            if (isset($_POST['modelParams'])) {
                $model_params = array_map('sanitize_text_field', $_POST['modelParams']);
            }
            if (isset($_POST['contentParams'])) {
                $content_params = array_map('sanitize_text_field', $_POST['contentParams']);
            }

            $output = '';

            try {

                if (empty($content_params['prompt'])) {
                    throw new \Exception(esc_html__('You have not provided the topic for the content that you want to generate!', 'livemesh-el-assistant'));
                }

                $prompt = 'Write me content for the topic and instructions that follow: ' . $content_params['prompt'] . '\n';;
                if ($content_params['keywords'] != '') {
                    $prompt .= 'Place these keywords: ' . $content_params['keywords'] . '\n';
                }
                $prompt .= 'Generate in this language: ' . $content_params['language'] . '\n';
                $prompt .= 'Writing Style: ' . $content_params['writingStyle'] . '\n';
                $prompt .= 'Writing Tone: ' . $content_params['writingTone'] . '\n';

                $call_params = array(
                    'model' => $model_params['gpt3Model'],
                    'max_tokens' => intval($model_params['maxTokens']),
                    'temperature' => floatval($model_params['temperature']),
                    'top_p' => 1,
                    'n' => 1,
                    'stream' => false,
                    'frequency_penalty' => floatval($model_params['frequencyPenalty']),
                    'presence_penalty' => floatval($model_params['presencePenalty']),
                );

                $chat_mode = false;
                if($call_params['model'] == 'gpt-3.5-turbo' || $call_params['model'] == 'gpt-4' || $call_params['model'] == 'gpt-4-32k') {
                    $chat_mode = true;
                }

                if ($chat_mode) {
                    $url = 'https://api.openai.com/v1/chat/completions';
                    $call_params['messages'] = array(
                        array('role' => 'user', 'content' => $prompt)
                    );
                }
                else {
                    $url = 'https://api.openai.com/v1/completions';
                    $call_params['prompt'] = $prompt;
                    $call_params['best_of'] = 1;
                }

                $response = $this->openai_client->dispatch($url, $call_params);

                $choices = $response['choices'];

                if ( count( $choices ) == 0 ) {
                    throw new \Exception(esc_html__('No response was generated. Please try again using different prompt!', 'livemesh-el-assistant'));
                }

                if ($chat_mode) {
                    $output = $choices[0]['message']['content'];
                }
                else {
                    $output =$choices[0]['text'];
                }

                $image = $this->generate_image();
                // If there is an error during image generation, the error message will be shown at the start of the content
                wp_send_json_success($image . trim($output));

            } catch (\Throwable $throwable) {
                wp_send_json_error(esc_html__('Error! ', 'livemesh-el-assistant') . $throwable->getMessage());
            }

        }

        /**
         * @throws \Exception
         */
        public function generate_image() {

            $image_params = array();

            if (isset($_POST['imageParams'])) {
                $image_params = array_map('sanitize_text_field', $_POST['imageParams']);
            }

            if (empty($image_params['imagePrompt'])) {
                return;
            }

            $output = '';

            // For now, let the prompt be the title for the image
            $image_alt = $image_params['imagePrompt'];
            $prompt = 'Generate an image of ' . $image_params['imagePrompt'] . '\n';
            if ($image_params['imageStyle'] != 'None') {
                $prompt .= 'Image Style: ' . $image_params['imageStyle'] . '\n';
            }

            $url = 'https://api.openai.com/v1/images/generations';

            $call_params = array(
                'prompt' => $prompt,
                'n' => 1,
                'size' => $image_params['imageSize'],
                'response_format' => 'url',
            );

            $response = $this->openai_client->dispatch($url, $call_params);

            $image_url = isset($response['data'][0]['url']) ? $response['data'][0]['url'] : '';

            if (!function_exists('media_sideload_image')) {
                require_once(ABSPATH . 'wp-admin/includes/media.php');
                require_once(ABSPATH . 'wp-admin/includes/file.php');
                require_once(ABSPATH . 'wp-admin/includes/image.php');
            }
            // Save to the media library and return me image html
            $image_src = media_sideload_image($image_url, 0, null, 'src');
            if (is_wp_error($image_src)) {
                throw new \Exception($image_src->get_error_message());
            }

            return '<img class="alignleft" src="' . $image_src . '" alt="' . $image_alt . '" />';

        }

    }

endif; // End if class_exists check


